<?php

namespace App\Http\Controllers\Auth;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;

use Illuminate\Http\Request;
use Inertia\Inertia;

class LoginController extends Controller
{
    public function create(){
        return Inertia::render('auth/Login');
    }

    public function create_forgot(){
        return Inertia::render('auth/ForgotPassword');
    }

    public function store(Request $request)
    {
        $credentials = $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);
        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();
            return redirect()->intended('/admin/dashboard');
        }
        return back()->withErrors([
            'message' => 'Invalid credentials. Please check email and password and try again',
        ]);
        dd('This should show on authentication failure.');
    }

    public function create_reset(Request $request, $token)
    {
        return Inertia::render('auth/Reset', [
            'token' => $token,
            'email' => $request->email, 
        ]);
    }

    public function store_forgot(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $status = Password::sendResetLink(
            $request->only('email')
        );

        return $status === Password::RESET_LINK_SENT
            ? redirect()
            ->route('confirm_send')
            ->with('status', __($status))
            : back()->withErrors(['email' => __($status)]);
    }

    public function update(Request $request)
    {
        // Validate
        $request->validate([
            'token'                 => ['required'],
            'email'                 => ['required', 'email'],
            'password'              => ['required', 'confirmed', 'min:8'],
        ]);

        // Attempt reset
        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->forceFill([
                    // Use bcrypt() directly
                    'password'       => bcrypt($password),
                    'remember_token' => Str::random(60),
                ])->save();
            }
        );

        // Redirect on outcome
        if ($status === Password::PASSWORD_RESET) {
            return redirect()
                ->route('login')
                ->with('status', __($status));
        }

        return back()->withErrors(['email' => __($status)]);
    }

    public function destroy()
    {
        Auth::logout();
        return redirect()->route('login');
    }
}
