<?php

namespace App\Http\Controllers;

use App\Models\Faq;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use DB;
use App\Traits\DeleteModelTrait; 

class FaqController extends Controller
{
    use DeleteModelTrait;
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $faqs = FAQ::orderBy('created_at', 'asc')->get()->map(function ($faq) {
            return [
                'question' => $faq->question,
                'answer'   => $faq->answer,
            ];
        });

        return Inertia::render('Faq', [
            'faqs' => $faqs,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        try{
            DB::beginTransaction();
            //validating form data
            $validatedData = $request->validate([
                'question' => 'required|min:3', 
                'status' => 'required|boolean',
                'answer' =>'required',
            
            ]);

            $faq = new Faq;
            $faq->question = $validatedData['question'];
            $faq->answer = $validatedData['answer'];
            $faq->status = $validatedData['status'];
            $faq->save();

            DB::commit();
            return to_route('admin.faqs')->with('success', [
                'message' => 'Faq created successfully',
                'check' => false
            ]);
        } catch (ValidationException $e){
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Validation Error: '.$e->getMessage(),
                'check' => $e->getMessage()
            ]);
        } catch(\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Error: '.$e->getMessage(),
                'check' => $e->getMessage()
            ]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Faq $faq)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $faq = Faq::findOrFail($id);
        return response()->json($faq);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $validatedData = $request->validate([
                'question' => 'required|min:3', 
                'answer'=>'required|min:5',
                'status' => 'required|boolean',
                
            ]);

            // Retrieve the faq or fail.
            $faq = Faq::findOrFail($id);
            $faq->fill($validatedData);
            if ($faq->isDirty()){
                $faq->save();
            }

            DB::commit();

            return Redirect::back()->with('success', [
                'message' => 'FAQ updated successfully',
                'check'   => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Validation error: ' . $e->getMessage(),
                'check'   => $e->getMessage()
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Error: ' . $e->getMessage(),
                'check'   => $e->getMessage()
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        return $this->deleteModel($id, new Faq());
    }

    public function fetchFaqs()
    {
        // Fetch all faqs
        $faqs = Faq::orderBy('created_at', 'desc')->get();
        return Inertia::render('faqs/Faq', [
            'faqs' => $faqs->map(function ($faq) {
                return $faq->only('id','question', 'answer' ,'status','created_at');
            }),
        ]);
    }
}
