<?php

namespace App\Http\Controllers;

use App\Models\Banner;
use App\Models\Slider;
use App\Models\Testimonial;
use App\Models\MenuItem;
use App\Models\Gallary;
use App\Models\Accommodation;
use App\Models\EventRoom;
use App\Models\Service;
use App\Models\HomepageVideo;
use Inertia\Inertia;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    public function index()
    {
        $banner = Banner::where('status', true)->first();
        $slider = Slider::where('status', true)->first();
        $testimonials = Testimonial::where('status', true)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($testimonial) {
                return [
                    'id' => $testimonial->id,
                    'name' => $testimonial->name,
                    'message' => $testimonial->message,
                    'photo' => $testimonial->photo,
                ];
            });

        // Get menu items for home page menu section
        $menuItems = MenuItem::where('status', true)
            ->orderBy('display_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->limit(6) // Limit to 6 items for home page preview
            ->get();

        $grouped = [
            'Food' => [],
            'Drinks' => [],
            'Desserts' => [],
        ];

        foreach ($menuItems as $item) {
            $category = $item->category ?? null;
            if ($category && isset($grouped[$category]) && count($grouped[$category]) < 2) {
                $grouped[$category][] = [
                    'id' => $item->id,
                    'name' => $item->name,
                    'price' => $item->price_usd ?? $item->price_fbu ?? $item->price_eur,
                    'price_fbu' => $item->price_fbu,
                    'price_usd' => $item->price_usd,
                    'price_eur' => $item->price_eur,
                    'description' => $item->description,
                ];
            }
        }

        // Get gallery images for photo slider
        $gallery = Gallary::where('status', true)
            ->select('photo', 'title', 'caption')
            ->orderBy('created_at', 'desc')
            ->get();

        // Get accommodations for home page rooms section (limit to 3)
        $accommodations = Accommodation::where('status', true)
            ->orderBy('created_at', 'desc')
            ->limit(3)
            ->get()
            ->map(function ($accommodation) {
                return [
                    'id' => $accommodation->id,
                    'room_name' => $accommodation->room_name,
                    'room_category' => $accommodation->room_category,
                    'primary_photo' => $accommodation->primary_photo,
                    'price_fbu' => $accommodation->price_fbu,
                    'price_usd' => $accommodation->price_usd,
                    'price_eur' => $accommodation->price_eur,
                ];
            });

        // Get event rooms for home page events section (limit to 3)
        $eventRooms = EventRoom::where('status', true)
            ->orderBy('created_at', 'desc')
            ->limit(3)
            ->get()
            ->map(function ($eventRoom) {
                return [
                    'id' => $eventRoom->id,
                    'name' => $eventRoom->name,
                    'primary_photo' => $eventRoom->primary_photo,
                    'price_fbu' => $eventRoom->price_fbu,
                    'price_usd' => $eventRoom->price_usd,
                    'price_eur' => $eventRoom->price_eur,
                    'short_description' => $eventRoom->short_description,
                    'capacity' => $eventRoom->capacity,
                    'layouts' => $eventRoom->layouts ? (is_array($eventRoom->layouts) ? $eventRoom->layouts : json_decode($eventRoom->layouts, true)) : [],
                    'created_at' => $eventRoom->created_at,
                ];
            });

        // Get services for home page services section
        $services = Service::where('status', true)
            ->orderBy('display_order', 'asc')
            ->orderBy('created_at', 'asc')
            ->get()
            ->map(function ($service) {
                return [
                    'id' => $service->id,
                    'name' => $service->name,
                    'description' => $service->description,
                    'photo' => $service->photo,
                    'icon' => $service->icon,
                    'link' => $service->link,
                    'display_order' => $service->display_order,
                    'status' => $service->status,
                ];
            });

        // Get homepage videos for home page videos section
        $videos = HomepageVideo::where('status', true)
            ->orderBy('display_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($video) {
                return [
                    'id' => $video->id,
                    'title' => $video->title,
                    'description' => $video->description,
                    'video_path' => $video->video_path ? "/storage/{$video->video_path}" : null,
                    'video_url' => $video->video_url,
                    'status' => $video->status,
                ];
            });
        
        return Inertia::render('Home', [
            'banner' => $banner,
            'slider' => $slider,
            'testimonials' => $testimonials,
            'menuFoodItems' => $grouped['Food'],
            'menuDrinkItems' => $grouped['Drinks'],
            'menuDessertItems' => $grouped['Desserts'],
            'gallery' => $gallery,
            'accommodations' => $accommodations,
            'eventRooms' => $eventRooms,
            'services' => $services,
            'videos' => $videos,
        ]);
    }

    public function rooms()
    {
        // Get all accommodations for rooms listing
        $accommodations = Accommodation::where('status', true)
            ->orderBy('room_name')
            ->get()
            ->map(function ($accommodation) {
                return [
                    'id' => $accommodation->id,
                    'room_name' => $accommodation->room_name,
                    'room_category' => $accommodation->room_category,
                    'primary_photo' => $accommodation->primary_photo,
                    'price_fbu' => $accommodation->price_fbu,
                    'price_usd' => $accommodation->price_usd,
                    'price_eur' => $accommodation->price_eur,
                    'description' => $accommodation->description,
                ];
            });

        // Get featured accommodations (where featured is true)
        $featuredAccommodations = Accommodation::where('status', true)
            ->where('featured', true)
            ->orderBy('room_name')
            ->limit(2) // Limit to 2 for the featured section
            ->get()
            ->map(function ($accommodation) {
                return [
                    'id' => $accommodation->id,
                    'room_name' => $accommodation->room_name,
                    'room_category' => $accommodation->room_category,
                    'primary_photo' => $accommodation->primary_photo,
                    'price_fbu' => $accommodation->price_fbu,
                    'price_usd' => $accommodation->price_usd,
                    'price_eur' => $accommodation->price_eur,
                    'description' => $accommodation->description,
                ];
            });
        
        return Inertia::render('Rooms', [
            'accommodations' => $accommodations,
            'featuredAccommodations' => $featuredAccommodations,
        ]);
    }

    public function reservation()
    {
        $accommodations = Accommodation::where('status', true)
            ->with('pricingRule')
            ->orderBy('room_name')
            ->get()
            ->map(function ($accommodation) {
                return [
                    'id' => $accommodation->id,
                    'room_name' => $accommodation->room_name,
                    'room_category' => $accommodation->room_category,
                    'price_fbu' => $accommodation->price_fbu,
                    'price_usd' => $accommodation->price_usd,
                    'price_eur' => $accommodation->price_eur,
                    'pricing_rule' => $accommodation->pricingRule ? [
                        'base_price_bif' => $accommodation->pricingRule->base_price_bif,
                        'base_price_usd' => $accommodation->pricingRule->base_price_usd,
                        'base_price_eur' => $accommodation->pricingRule->base_price_eur,
                        'additional_adult_bif' => $accommodation->pricingRule->additional_adult_bif,
                        'additional_adult_usd' => $accommodation->pricingRule->additional_adult_usd,
                        'additional_adult_eur' => $accommodation->pricingRule->additional_adult_eur,
                        'child_surcharge_bif' => $accommodation->pricingRule->child_surcharge_bif,
                        'child_surcharge_usd' => $accommodation->pricingRule->child_surcharge_usd,
                        'child_surcharge_eur' => $accommodation->pricingRule->child_surcharge_eur,
                    ] : null,
                ];
            });

        $eventRooms = EventRoom::where('status', true)
            ->orderBy('name')
            ->get()
            ->map(function ($eventRoom) {
                return [
                    'id' => $eventRoom->id,
                    'name' => $eventRoom->name,
                    'price_fbu' => $eventRoom->price_fbu,
                    'price_usd' => $eventRoom->price_usd,
                    'price_eur' => $eventRoom->price_eur,
                ];
            });

        return Inertia::render('Reservation', [
            'accommodations' => $accommodations,
            'eventRooms' => $eventRooms,
        ]);
    }

    public function about()
    {
        // Get featured accommodations
        $featuredAccommodations = Accommodation::where('status', true)
            ->where('featured', true)
            ->orderBy('room_name')
            ->limit(2)
            ->get()
            ->map(function ($accommodation) {
                return [
                    'id' => $accommodation->id,
                    'room_name' => $accommodation->room_name,
                    'room_category' => $accommodation->room_category,
                    'primary_photo' => $accommodation->primary_photo,
                    'price_fbu' => $accommodation->price_fbu,
                    'price_usd' => $accommodation->price_usd,
                    'price_eur' => $accommodation->price_eur,
                    'description' => $accommodation->description,
                ];
            });

        // Get featured event rooms
        $featuredEventRooms = EventRoom::where('status', true)
            ->orderBy('name')
            ->limit(2)
            ->get()
            ->map(function ($eventRoom) {
                return [
                    'id' => $eventRoom->id,
                    'name' => $eventRoom->name,
                    'primary_photo' => $eventRoom->primary_photo,
                    'price_fbu' => $eventRoom->price_fbu,
                    'price_usd' => $eventRoom->price_usd,
                    'price_eur' => $eventRoom->price_eur,
                    'short_description' => $eventRoom->short_description,
                ];
            });

        return Inertia::render('AboutUs', [
            'featuredAccommodations' => $featuredAccommodations,
            'featuredEventRooms' => $featuredEventRooms,
        ]);
    }
}
