<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Accommodation;
use App\Models\Room;
use App\Models\PricingRule;
use Illuminate\Support\Facades\DB;

class RoomSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * Room data specification from client requirements:
     * Room No | Category | Price BIF | Price USD | Price EUR
     */
    public function run(): void
    {
        // Define room data: [room_number, category, price_bif, price_usd, price_eur]
        $roomData = [
            // Simple category
            ['101', 'Simple', 100000, 40, 35],
            ['102', 'Simple', 100000, 40, 35],
            ['103', 'Simple', 100000, 40, 35],
            ['104', 'Simple', 100000, 40, 35],
            
            // Silver category (different prices)
            ['105', 'Silver', 180000, 65, 60],
            ['110', 'Silver', 130000, 65, 60],
            
            // Golden category
            ['106', 'Golden', 120000, 50, 45],
            ['201', 'Golden', 120000, 50, 45],
            ['202', 'Golden', 120000, 50, 45],
            ['203', 'Golden', 120000, 50, 45],
            ['204', 'Golden', 120000, 50, 45],
            ['205', 'Golden', 120000, 50, 45],
            ['206', 'Golden', 120000, 50, 45],
            ['207', 'Golden', 120000, 50, 45],
            ['208', 'Golden', 120000, 50, 45],
            ['209', 'Golden', 120000, 50, 45],
            ['210', 'Golden', 120000, 50, 45],
            ['211', 'Golden', 120000, 50, 45],
            ['212', 'Golden', 120000, 50, 45],
            ['213', 'Golden', 120000, 50, 45],
            ['214', 'Golden', 120000, 50, 45],
            ['215', 'Golden', 120000, 50, 45],
            ['216', 'Golden', 120000, 50, 45],
            ['217', 'Golden', 120000, 50, 45],
            ['301', 'Golden', 120000, 50, 45],
            ['302', 'Golden', 120000, 50, 45],
            ['303', 'Golden', 120000, 50, 45],
            ['304', 'Golden', 120000, 50, 45],
            ['305', 'Golden', 120000, 50, 45],
            
            // Standard category
            ['111', 'Standard', 110000, 45, 45],
            ['112', 'Standard', 110000, 45, 45],
            ['113', 'Standard', 110000, 45, 45],
            ['114', 'Standard', 110000, 45, 45],
            ['115', 'Standard', 110000, 45, 45],
            ['116', 'Standard', 110000, 45, 45],
            
            // VIP category (different prices)
            ['117', 'VIP', 150000, 75, 70],
            ['218', 'VIP', 200000, 75, 70],
            ['307', 'VIP', 200000, 75, 70],
        ];

        DB::transaction(function () use ($roomData) {
            // Track accommodations by category for pricing rules
            $accommodationsByCategory = [];

            foreach ($roomData as [$roomNumber, $category, $priceBif, $priceUsd, $priceEur]) {
                // Find or create accommodation category (use lowest price as base)
                if (!isset($accommodationsByCategory[$category])) {
                    $accommodation = Accommodation::firstOrCreate(
                        ['room_category' => $category],
                        [
                            'room_name' => $category,
                            'room_category' => $category,
                            'price_fbu' => $priceBif, // Will be updated to lowest if multiple prices exist
                            'price_usd' => $priceUsd,
                            'price_eur' => $priceEur,
                            'description' => "Comfortable {$category} room with modern amenities",
                            'full_description' => "Our {$category} room offers a perfect blend of comfort and functionality, ideal for both business and leisure travelers.",
                            'status' => true,
                            'featured' => false,
                        ]
                    );
                    $accommodationsByCategory[$category] = $accommodation;
                } else {
                    $accommodation = $accommodationsByCategory[$category];
                    // Update to lowest price if this room has lower price
                    if ($priceBif < $accommodation->price_fbu) {
                        $accommodation->update([
                            'price_fbu' => $priceBif,
                            'price_usd' => $priceUsd,
                            'price_eur' => $priceEur,
                        ]);
                    }
                }

                // Create individual room
                Room::firstOrCreate(
                    [
                        'room_number' => $roomNumber,
                        'accommodation_id' => $accommodation->id,
                    ],
                    ['is_available' => true]
                );
            }

            // Create pricing rules for each category (one rule per category)
            foreach ($accommodationsByCategory as $category => $accommodation) {
                PricingRule::updateOrCreate(
                    ['accommodation_id' => $accommodation->id],
                    [
                        'base_price_bif' => $accommodation->price_fbu,
                        'base_price_usd' => $accommodation->price_usd,
                        'base_price_eur' => $accommodation->price_eur,
                        'additional_adult_bif' => 0, // Default: no additional charge (can be configured in admin)
                        'additional_adult_usd' => 0,
                        'additional_adult_eur' => 0,
                        'child_surcharge_bif' => 20000, // Default: 20,000 BIF per child (can be configured)
                        'child_surcharge_usd' => 10,
                        'child_surcharge_eur' => 8,
                    ]
                );
            }
        });
    }
}
