import React, { useState, useEffect } from "react";
import axios from "axios";
import { toast } from "react-hot-toast";

const FileUploadInput = ({
  label = "Upload", // Label text
  inputId = "inputGroupFile01", // Unique ID for the input
  buttonText = "Choose file", // Default button text
  uploadUrl, // API endpoint for the upload
  onSuccess, // Callback for successful upload
  onError, // Callback for handling errors
  previewUrl = "", // Optional prop for preset image preview
  helpText = "", // Optional help text to display below the input
}) => {
  const [fileName, setFileName] = useState(buttonText); // State for the file label
  const [preview, setPreview] = useState(previewUrl); // State for the image preview

  // Ensure the preset previewUrl is used on initial render
  useEffect(() => {
    setPreview(previewUrl);
  }, [previewUrl]);

  const handleFileChange = async (e) => {
    const file = e.target.files[0];
    if (!file) return;

    // Show preview of the selected file
    const fileReader = new FileReader();
    fileReader.onload = () => setPreview(fileReader.result);
    fileReader.readAsDataURL(file);

    const formData = new FormData();
    formData.append("file", file);

    toast.promise(
      axios.post(uploadUrl, formData, {
        headers: { "Content-Type": "multipart/form-data" },
      }),
      {
        loading: "Uploading...",
        success: (response) => {
          const uploadedFileName = response.data.filename;
          const trimmedFileName =
            uploadedFileName.length > 20
              ? uploadedFileName.substring(0, 20) + "..."
              : uploadedFileName;

          // Update the fileName state to display the uploaded file name
          setFileName(trimmedFileName);

          // Trigger the onSuccess callback if provided
          if (onSuccess) onSuccess(uploadedFileName);

          return "Upload complete!";
        },
        error: (error) => {
          // Reset fileName to default text on error
          setFileName(buttonText);

          // Keep the preset preview if upload fails
          setPreview(previewUrl);

          // Trigger the onError callback if provided
          if (onError) onError(error);

          return "Upload failed!";
        },
      }
    );
  };

  return (
    <div className="mb-3">
      <div className="input-group">
        <div className="input-group-prepend">
          <span className="input-group-text">{label}</span>
        </div>
        <div className="custom-file">
          <input
            type="file"
            className="custom-file-input"
            id={inputId}
            onChange={handleFileChange}
          />
          <label className="custom-file-label" htmlFor={inputId}>
            {fileName}
          </label>
        </div>
      </div>

      {/* Show Image Preview if available */}
      {preview && (
        <div className="mt-3">
          <img
            src={preview}
            alt="File Preview"
            className="img-thumbnail"
            style={{ maxWidth: "150px", maxHeight: "150px" }}
          />
        </div>
      )}

      {/* Show help text if provided */}
      {helpText && (
        <small id={`${inputId}Help`} className="form-text text-muted">
          {helpText}
        </small>
      )}
    </div>
  );
};

export default FileUploadInput;
