import React, { useEffect, useRef, useState } from 'react';

const WysiwygEditor = ({
    value = '',
    onChange,
    height = 300,
    placeholder = '',
    toolbar = 'full',
    invalidFeedback,
    disabled = false,
    name = 'content'
}) => {
    const editorRef = useRef(null);
    const containerRef = useRef(null);
    const [isInitialized, setIsInitialized] = useState(false);
    const inputClass = `form-control ${invalidFeedback ? 'is-invalid' : ''}`;

    const toolbarOptions = {
        minimal: [
            ['style', ['style']],
            ['font', ['bold', 'italic', 'underline', 'clear']],
            ['para', ['ul', 'ol']],
            ['insert', ['link']]
        ],
        standard: [
            ['style', ['style']],
            ['font', ['bold', 'italic', 'underline', 'clear']],
            ['fontname', ['fontname']],
            ['para', ['ul', 'ol', 'paragraph']],
            ['insert', ['link', 'picture']]
        ],
        full: [
            ['style', ['style']],
            ['font', ['bold', 'italic', 'underline', 'clear']],
            ['fontname', ['fontname']],
            ['fontsize', ['fontsize']],
            ['color', ['color']],
            ['para', ['ul', 'ol', 'paragraph']],
            ['table', ['table']],
            ['insert', ['link', 'picture', 'video']],
            ['view', ['fullscreen', 'codeview']]
        ]
    };

    const initializeEditor = async () => {
        if (!window.jQuery || !window.jQuery.fn.summernote) {
            await loadDependencies();
        }

        const $editor = window.jQuery(editorRef.current);

        // Destroy existing instance if it exists
        if ($editor.summernote) {
            $editor.summernote('destroy');
        }

        // Initialize new instance
        $editor.summernote({
            height: height,
            placeholder: placeholder,
            toolbar: toolbarOptions[toolbar] || toolbarOptions.standard,
            callbacks: {
                onChange: function(contents) {
                    onChange(contents);
                },
                onInit: function() {
                    $editor.next().find('.note-editable').css({
                        'word-wrap': 'break-word',
                        'word-break': 'break-word',
                        'overflow-wrap': 'break-word',
                        'white-space': 'pre-wrap'
                    });

                    // Set initial content
                    if (value) {
                        $editor.summernote('code', value);
                    }

                    setIsInitialized(true);
                }
            },
            styleTags: ['p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6'],
            dialogsInBody: true,
            tabDisable: false
        });

        if (disabled) {
            $editor.summernote('disable');
        }
    };

    const loadDependencies = async () => {
        // Load Summernote (Updated to 0.8.20)
        if (!document.querySelector('link[href*="summernote"]')) {
            await loadCSS('https://cdnjs.cloudflare.com/ajax/libs/summernote/0.8.20/summernote-bs4.min.css');
        }

        if (!window.jQuery.fn.summernote) {
            await loadScript('https://cdnjs.cloudflare.com/ajax/libs/summernote/0.8.20/summernote-bs4.min.js');
        }

        addCustomStyles();
    };

    const loadScript = (src) => {
        return new Promise((resolve, reject) => {
            const script = document.createElement('script');
            script.src = src;
            script.onload = resolve;
            script.onerror = () => reject(new Error(`Failed to load script: ${src}`));
            document.head.appendChild(script);
        });
    };

    const loadCSS = (href) => {
        return new Promise((resolve, reject) => {
            const link = document.createElement('link');
            link.rel = 'stylesheet';
            link.href = href;
            link.onload = resolve;
            link.onerror = () => reject(new Error(`Failed to load CSS: ${href}`));
            document.head.appendChild(link);
        });
    };

    const addCustomStyles = () => {
        if (!document.querySelector('#summernote-fixes')) {
            const styleSheet = document.createElement('style');
            styleSheet.id = 'summernote-fixes';
            styleSheet.textContent = `
                .note-editor.note-frame {
                    border: 1px solid #ced4da;
                }
                .note-editor.note-frame .note-toolbar {
                    background-color: #f8f9fa;
                    border-bottom: 1px solid #ced4da;
                    padding: 5px 10px;
                    display: flex;
                    flex-wrap: wrap;
                    gap: 5px;
                    align-items: center;
                }
                .note-editor .note-btn-group {
                    margin: 0;
                    display: inline-flex;
                    gap: 2px;
                }
                .note-editor .note-btn {
                    background-color: white;
                    border: 1px solid #ced4da;
                    padding: 5px 10px;
                    font-size: 14px;
                    color: #495057;
                    border-radius: 3px;
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                }
                .note-editable {
                    word-wrap: break-word !important;
                    word-break: break-word !important;
                    overflow-wrap: break-word !important;
                    white-space: pre-wrap !important;
                    width: 100% !important;
                }
                .note-editing-area {
                    position: relative !important;
                    width: 100% !important;
                }
                .note-frame {
                    width: 100% !important;
                }
                .note-editable table {
                    width: 100% !important;
                    table-layout: fixed !important;
                }
                .note-editable table td,
                .note-editable table th {
                    word-wrap: break-word !important;
                    overflow-wrap: break-word !important;
                }
                .note-editable img {
                    max-width: 100% !important;
                    height: auto !important;
                }
            `;
            document.head.appendChild(styleSheet);
        }
    };

    useEffect(() => {
        initializeEditor();

        return () => {
            const $editor = window.jQuery && window.jQuery(editorRef.current);
            if ($editor && $editor.summernote) {
                $editor.summernote('destroy');
            }
            setIsInitialized(false);
        };
    }, []);

    useEffect(() => {
        if (isInitialized && window.jQuery) {
            const $editor = window.jQuery(editorRef.current);
            if ($editor.summernote && value !== $editor.summernote('code')) {
                $editor.summernote('code', value);
            }
        }
    }, [value, isInitialized]);

    return (
        <div ref={containerRef} className="wysiwyg-container">
            <div ref={editorRef} name={name} />
            <input type="hidden" name={name} value={value} />
        </div>
    );
};

export default WysiwygEditor;