import Col from "@admin/ui/Col";
import Row from "@admin/ui/Row";
import StyledTable from "@admin/ui/StyledTable";
import { useState } from "react";
import DeleteConfirmationModal from "@admin/ui/DeleteConfirmationModal";
import { toast, Toaster } from "react-hot-toast";
import CreateEditMenuItem from "./CreateEditMenuItem";
import { formatPriceTriple } from "@/utils/priceFormatter";

export default function Menu({ foodItems, drinkItems, dessertItems }) {
    const [activeTab, setActiveTab] = useState('food');
    const [isEditModalOpen, setIsEditModalOpen] = useState(false);
    const [initialData, setInitialData] = useState(null);
    const [deleteModal, setDeleteModal] = useState({ show: false, menuItem: null });

    const columns = [
        { header: '', key: 'checkbox' },
        { header: 'Name', key: 'name' },
        { header: 'Category', key: 'category' },
        { header: 'Price', key: 'price' },
        { header: 'Status', key: 'status' },
    ];

    const withDisplayPrices = (items) => (items || []).map((item) => ({
        ...item,
        price: formatPriceTriple(item.price_fbu, item.price_usd, item.price_eur) || 'Price on request',
    }));

    const handleDelete = (row) => {
        setDeleteModal({ show: true, menuItem: row });
    };

    const handleEdit = (row) => {
        fetch(`/admin/edit_menu_item/${row.id}`)
            .then((response) => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then((menuItemData) => {
                setInitialData(menuItemData);
                setIsEditModalOpen(true);
            })
            .catch((error) => {
                console.error('Error fetching menu item details:', error);
                toast.error('Failed to load menu item details');
            });
    };

    const handleCreate = () => {
        setInitialData(null);
        setIsEditModalOpen(true);
    };

    const handleCloseModal = () => {
        setIsEditModalOpen(false);
        setInitialData(null);
    };

    return (
        <>
            <Toaster />
            <Row md="12">
                <Col lg="12">
                    <h2 className="page-title">Menu Items</h2>
                    <p className="text-muted">Manage restaurant menu items by category</p>
                    <button className="btn mb-4 btn-primary" onClick={handleCreate}>
                        Create New Menu Item
                    </button>

                    {/* Bootstrap Tabs */}
                    <ul className="nav nav-tabs mb-4" id="menuTabs" role="tablist">
                        <li className="nav-item">
                            <a
                                className={`nav-link ${activeTab === 'food' ? 'active' : ''}`}
                                onClick={() => setActiveTab('food')}
                                role="tab"
                                style={{ cursor: 'pointer' }}
                            >
                                Food
                            </a>
                        </li>
                        <li className="nav-item">
                            <a
                                className={`nav-link ${activeTab === 'drinks' ? 'active' : ''}`}
                                onClick={() => setActiveTab('drinks')}
                                role="tab"
                                style={{ cursor: 'pointer' }}
                            >
                                Drinks
                            </a>
                        </li>
                        <li className="nav-item">
                            <a
                                className={`nav-link ${activeTab === 'desserts' ? 'active' : ''}`}
                                onClick={() => setActiveTab('desserts')}
                                role="tab"
                                style={{ cursor: 'pointer' }}
                            >
                                Desserts
                            </a>
                        </li>
                    </ul>

                    {/* Tab Content */}
                    <div className="tab-content">
                        <div className={`tab-pane fade ${activeTab === 'food' ? 'show active' : ''}`} role="tabpanel">
                            <Row>
                                <Col lg="12">
                                    <StyledTable
                                        data={withDisplayPrices(foodItems)}
                                        columns={columns}
                                        itemsPerPage={10}
                                        title="Food Items"
                                        onEdit={handleEdit}
                                        onDelete={handleDelete}
                                        noDataText="No food items found"
                                    />
                                </Col>
                            </Row>
                        </div>

                        <div className={`tab-pane fade ${activeTab === 'drinks' ? 'show active' : ''}`} role="tabpanel">
                            <Row>
                                <Col lg="12">
                                    <StyledTable
                                        data={withDisplayPrices(drinkItems)}
                                        columns={columns}
                                        itemsPerPage={10}
                                        title="Drink Items"
                                        onEdit={handleEdit}
                                        onDelete={handleDelete}
                                        noDataText="No drink items found"
                                    />
                                </Col>
                            </Row>
                        </div>

                        <div className={`tab-pane fade ${activeTab === 'desserts' ? 'show active' : ''}`} role="tabpanel">
                            <Row>
                                <Col lg="12">
                                    <StyledTable
                                        data={withDisplayPrices(dessertItems)}
                                        columns={columns}
                                        itemsPerPage={10}
                                        title="Dessert Items"
                                        onEdit={handleEdit}
                                        onDelete={handleDelete}
                                        noDataText="No dessert items found"
                                    />
                                </Col>
                            </Row>
                        </div>
                    </div>
                </Col>
            </Row>

            <DeleteConfirmationModal
                isOpen={deleteModal.show}
                onClose={() => setDeleteModal({ show: false, menuItem: null })}
                onConfirmDelete={() => {
                    toast.success("Menu Item Deleted Successfully");
                }}
                itemName={deleteModal.menuItem?.name || 'this menu item'}
                itemType="menu item"
                deleteEndpoint={`/admin/api/delete_menu_item/${deleteModal?.menuItem?.id}`}
            />

            <CreateEditMenuItem
                isOpen={isEditModalOpen}
                onClose={handleCloseModal}
                initialData={initialData}
                defaultCategory={activeTab === 'food' ? 'Food' : activeTab === 'drinks' ? 'Drinks' : 'Desserts'}
            />
        </>
    );
}

