import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import Modal from "@admin/ui/Modal";
import Row from "@admin/ui/Row";
import Col from "@admin/ui/Col";
import { useEffect } from "react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { useForm } from "@inertiajs/react";

export default function EditRoom({ isOpen, onClose, initialData, accommodations = [] }) {
    const { data, setData, put, processing, errors } = useForm({
        'room_number': '',
        'accommodation_id': '',
        'is_available': true,
    });

    useEffect(() => {
        if (initialData) {
            setData({
                room_number: initialData?.room_number || '',
                accommodation_id: initialData?.accommodation_id || '',
                is_available: initialData?.is_available ?? true,
            });
        }
    }, [initialData, setData]);

    const handleSave = (event) => {
        event.preventDefault();

        const formResponse = useFormResponse(() => {
            onClose?.();
        });

        put(`/admin/api/update_room/${initialData.id}`, {
            preserveScroll: true,
            onSuccess: formResponse.onSuccess,
            onError: formResponse.onError,
        });
    };

    return (
        <Modal 
            isOpen={isOpen} 
            onClose={onClose} 
            title={`Edit Room ${initialData?.room_number || ''}`} 
            size="medium"
            primaryButtonText="Save Changes"
            onPrimaryAction={handleSave}
            processing={processing}
        >
            <form onSubmit={handleSave}>
                <Row>
                    <Col md="12">
                        <FormGroup
                            id="room_number"
                            label="Room Number"
                            type="text"
                            value={data.room_number}
                            onChange={(e) => setData("room_number", e.target.value)}
                            required
                            invalidFeedback={errors.room_number}
                            helpText="Enter the room number (e.g., 101, 102, 201)"
                        />

                        <FormGroup
                            id="accommodation_id"
                            label="Room Category"
                            as="select"
                            value={data.accommodation_id}
                            onChange={(e) => setData("accommodation_id", e.target.value)}
                            required
                            invalidFeedback={errors.accommodation_id}
                            helpText="Select the accommodation category this room belongs to"
                        >
                            <option value="">Select Category</option>
                            {accommodations.map((accommodation) => (
                                <option key={accommodation.id} value={accommodation.id}>
                                    {accommodation.name}
                                </option>
                            ))}
                        </FormGroup>

                        <div className="form-group">
                            <Switch
                                id="is_available"
                                label="Room Available"
                                checked={data.is_available}
                                onChange={(checked) => setData('is_available', checked)}
                                helpText="Toggle room availability status"
                            />
                        </div>
                    </Col>
                </Row>
            </form>
        </Modal>
    );
}



