import { Link, usePage } from '@inertiajs/react';
import { formatPriceTriple } from '@/utils/priceFormatter';

export default function Rooms({ accommodations = [] }) {
    const { siteinfo } = usePage().props;
    
    // Get dynamic description
    const roomsDescription = siteinfo?.rooms_description || '';
    // Generate room items - use accommodations if available, otherwise use 3 placeholders
    const getRoomItems = () => {
        if (accommodations && accommodations.length > 0) {
            // Take up to 3 accommodations
            return accommodations.slice(0, 3).map((acc) => ({
                id: acc.id != null ? String(acc.id) : '',
                name: acc.room_name,
                image: acc.primary_photo ? `/storage/${acc.primary_photo}` : 'https://placehold.co/800x571',
                price: formatPriceTriple(acc.price_fbu, acc.price_usd, acc.price_eur) || 'Price on request',
            }));
        }
        
        // Return 3 placeholder items if no accommodations
        return Array.from({ length: 3 }, (_, index) => ({
            id: `placeholder-${index + 1}`,
            name: `Room ${index + 1}`,
            image: 'https://placehold.co/800x571',
            price: 'Price on request',
        }));
    };

    const roomItems = getRoomItems();

    return (
        <>
        <section className="section">
        <div className="container">
            <div className="row justify-content-center text-center mb-5">
            <div className="col-md-7">
                <h2 className="heading" data-aos="fade-up">Rooms &amp; Suites</h2>
                {roomsDescription && (
                    <p data-aos="fade-up" data-aos-delay="100">{roomsDescription}</p>
                )}
            </div>
            </div>
            <div className="row">
            {roomItems.map((room, index) => (
                <div key={room.id} className="col-md-6 col-lg-4" data-aos="fade-up" data-aos-delay={index * 100}>
                    <Link href={room.id && String(room.id).startsWith('placeholder') ? '#' : `/accommodations/${room.id}`} className="room">
                    <figure className="img-wrap">
                        <img src={room.image} alt={room.name} className="img-fluid mb-3" />
                    </figure>
                    <div className="p-3 text-center room-info">
                        <h2>{room.name}</h2>
                        <span className="text-uppercase letter-spacing-1">{room.price} / per night</span>
                    </div>
                    </Link>
                </div>
            ))}
            </div>
        </div>
        </section>
        </>
    );
}