import { Head, usePage, Link } from "@inertiajs/react";
import HeroSection from "@user/HeroSection";
import { formatPriceTriple } from "@/utils/priceFormatter";

export default function AccommodationDetail() {
    const { accommodation } = usePage().props;

    // Default/placeholder data if no accommodation is provided
    const room = accommodation || {
        id: 1,
        room_name: "Deluxe Suite",
        primary_photo: null,
        price_fbu: "150000.00",
        price_usd: "85.00",
        description: "Experience luxury and comfort in our beautifully appointed Deluxe Suite, featuring modern amenities and stunning views.",
        full_description: "Our Deluxe Suite offers an unparalleled experience of comfort and elegance. The spacious room features a king-size bed with premium linens, a comfortable seating area, and floor-to-ceiling windows that flood the space with natural light. The en-suite bathroom includes a rainfall shower, plush towels, and premium toiletries. Whether you're here for business or leisure, this suite provides the perfect retreat after a day of exploration.",
        size: "45 sqm",
        bed_type: "King Size",
        view: "City View",
        amenities: ["Free WiFi", "Air Conditioning", "Mini Bar", "Room Service", "Flat Screen TV", "Safe Box", "Coffee Maker", "Hair Dryer"],
        gallery: [],
        featured: true,
    };

    // Get primary photo or placeholder
    const getPrimaryPhoto = () => {
        if (room.primary_photo) {
            return `/storage/${room.primary_photo}`;
        }
        return 'https://placehold.co/800x571';
    };

    // Get secondary image (first from gallery or placeholder)
    const getSecondaryPhoto = () => {
        if (room.gallery && room.gallery.length > 0) {
            return `/storage/${room.gallery[0]}`;
        }
        return 'https://placehold.co/800x800';
    };

    // Get gallery items for Photo slider
    const getGalleryItems = () => {
        if (room.gallery && room.gallery.length > 0) {
            return room.gallery.map((photo, index) => ({
                image: `/storage/${photo}`,
                caption: `${room.room_name} - Image ${index + 1}`,
                alt: `${room.room_name} gallery image ${index + 1}`
            }));
        }
        
        // Return placeholders if no gallery
        return Array.from({ length: 4 }, (_, index) => ({
            image: `https://placehold.co/1123x700`,
            caption: `${room.room_name} - Image ${index + 1}`,
            alt: `Room gallery placeholder ${index + 1}`
        }));
    };

    const galleryItems = getGalleryItems();

    // Open lightbox for gallery images
    const openLightbox = (index) => {
        if (typeof window !== 'undefined' && window.$ && window.$.fancybox) {
            const items = galleryItems.map(item => ({
                src: item.image,
                opts: { caption: item.caption }
            }));
            window.$.fancybox.open(items, {}, index);
        }
    };

    // Format price display in triple currency (BIF | USD | EUR)
    const priceDisplay = formatPriceTriple(room.price_fbu, room.price_usd, room.price_eur);

    // Parse amenities if it's a string
    const amenities = Array.isArray(room.amenities) 
        ? room.amenities 
        : (room.amenities ? JSON.parse(room.amenities) : []);

    return (
        <>
            <Head title={room.room_name} />
            
            {/* Hero Section with Breadcrumbs */}
            <HeroSection heading={room.room_name} breadcrumbs={["Home", "Rooms", room.room_name]} />

            {/* Welcome-Style Intro Section */}
            <section className="py-5 bg-light">
                <div className="container">
                    <div className="row align-items-center">
                        {/* Images - Right Side */}
                        <div className="col-md-12 col-lg-7 ml-auto order-lg-2 position-relative mb-5" data-aos="fade-up">
                            <img src={getPrimaryPhoto()} alt={room.room_name} className="img-fluid rounded" />
                        </div>
                        
                        {/* Content - Left Side */}
                        <div className="col-md-12 col-lg-4 order-lg-1" data-aos="fade-up">
                            <h2 className="heading">{room.room_name}</h2>
                            <p className="mb-4">{room.description}</p>
                            
                            {/* Price Display */}
                            {priceDisplay && (
                                <p className="mb-3">
                                    <span style={{ fontSize: '1.4rem', fontWeight: 'bold', color: '#c9a227' }}>
                                        {priceDisplay}
                                    </span>
                                    <span className="text-muted d-block mt-1"> / night</span>
                                </p>
                            )}
                            
                            <p>
                                <Link href={`/reservation?room=${room.id}`} className="btn btn-primary text-white py-2 mr-3">Book Now</Link>
                                <span className="mr-3 font-family-serif"><em>or</em></span>
                                <a href="#room-gallery" className="text-uppercase letter-spacing-1">View Gallery</a>
                            </p>
                        </div>
                    </div>
                </div>
            </section>

            {/* Room Details Section */}
            <section className="section" id="room-details">
                <div className="container">
                    <div className="row justify-content-center text-center mb-5">
                        <div className="col-md-7">
                            <h2 className="heading" data-aos="fade-up">Room Details</h2>
                            <p data-aos="fade-up" data-aos-delay="100">
                                Everything you need to know about our {room.room_name}
                            </p>
                        </div>
                    </div>
                    
                    <div className="row">
                        {/* Room Specifications */}
                        <div className="col-lg-8" data-aos="fade-up">
                            <div className="bg-white p-4 p-md-5 mb-5 border rounded">
                                {/* Quick Info Cards */}
                                <div className="row mb-4">
                                    {room.size && (
                                        <div className="col-6 col-md-3 mb-3">
                                            <div className="text-center p-3 border rounded">
                                                <span className="icon-fullscreen d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                                <small className="text-muted d-block">Size</small>
                                                <strong>{room.size}</strong>
                                            </div>
                                        </div>
                                    )}
                                    {room.bed_type && (
                                        <div className="col-6 col-md-3 mb-3">
                                            <div className="text-center p-3 border rounded">
                                                <span className="icon-bed d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                                <small className="text-muted d-block">Bed Type</small>
                                                <strong>{room.bed_type}</strong>
                                            </div>
                                        </div>
                                    )}
                                    {room.view && (
                                        <div className="col-6 col-md-3 mb-3">
                                            <div className="text-center p-3 border rounded">
                                                <span className="icon-eye d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                                <small className="text-muted d-block">View</small>
                                                <strong>{room.view}</strong>
                                            </div>
                                        </div>
                                    )}
                                    <div className="col-6 col-md-3 mb-3">
                                        <div className="text-center p-3 border rounded">
                                            <span className="icon-check d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                            <small className="text-muted d-block">Status</small>
                                            <strong>{room.featured ? 'Featured' : 'Available'}</strong>
                                        </div>
                                    </div>
                                </div>

                                {/* Full Description */}
                                {room.full_description && (
                                    <div className="mb-4">
                                        <h4 className="mb-3">About This Room</h4>
                                        <p className="text-muted" style={{ lineHeight: '1.8' }}>
                                            {room.full_description}
                                        </p>
                                    </div>
                                )}

                                {/* Amenities */}
                                {amenities.length > 0 && (
                                    <div>
                                        <h4 className="mb-3">Amenities</h4>
                                        <div className="row">
                                            {amenities.map((amenity, index) => (
                                                <div key={index} className="col-6 col-md-4 mb-2">
                                                    <div className="d-flex align-items-center">
                                                        <span className="icon-check text-success mr-2"></span>
                                                        <span>{amenity}</span>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Booking Sidebar */}
                        <div className="col-lg-4" data-aos="fade-up" data-aos-delay="100">
                            <div className="bg-white p-4 mb-4 border rounded sticky-top" style={{ top: '100px' }}>
                                <h4 className="mb-4 text-center">Book This Room</h4>
                                
                                {/* Price Display */}
                                <div className="text-center mb-4 pb-4 border-bottom">
                                    {priceDisplay && (
                                        <div>
                                            <span style={{ fontSize: '1.8rem', fontWeight: 'bold', color: '#c9a227' }}>
                                                {priceDisplay}
                                            </span>
                                            <span className="text-muted d-block mt-1"> / night</span>
                                        </div>
                                    )}
                                </div>

                                {/* Room Highlights */}
                                <ul className="list-unstyled mb-4">
                                    {room.size && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            Room Size: {room.size}
                                        </li>
                                    )}
                                    {room.bed_type && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            {room.bed_type} Bed
                                        </li>
                                    )}
                                    {room.view && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            {room.view}
                                        </li>
                                    )}
                                    {amenities.length > 0 && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            {amenities.length} Amenities Included
                                        </li>
                                    )}
                                </ul>

                                {/* Book Now Button */}
                                <Link 
                                    href={`/reservation?room=${room.id}`} 
                                    className="btn btn-primary btn-block text-white py-3 font-weight-bold"
                                    style={{ fontSize: '1.1rem' }}
                                >
                                    Reserve Now
                                </Link>

                                <p className="text-center text-muted mt-3 mb-0" style={{ fontSize: '0.85rem' }}>
                                    Free cancellation available
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* Photo Gallery Section - Using Photo component structure */}
            <section className="section slider-section bg-light" id="room-gallery">
                <div className="container">
                    <div className="row justify-content-center text-center mb-5">
                        <div className="col-md-7">
                            <h2 className="heading" data-aos="fade-up">Room Gallery</h2>
                            <p data-aos="fade-up" data-aos-delay="100">
                                Take a closer look at our {room.room_name}. Browse through the images to see the room's features and ambiance.
                            </p>
                        </div>
                    </div>
                    <div className="row">
                        <div className="col-md-12">
                            <div className="home-slider major-caousel owl-carousel mb-5" data-aos="fade-up" data-aos-delay="200">
                                {galleryItems.map((item, index) => (
                                    <div key={index} className="slider-item">
                                        <img 
                                            src={item.image} 
                                            alt={item.alt} 
                                            className="img-fluid" 
                                            style={{ cursor: 'pointer' }}
                                            onClick={() => openLightbox(index)}
                                        />
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* CTA Section */}
            <section className="section">
                <div className="container">
                    <div className="row justify-content-center text-center">
                        <div className="col-md-8" data-aos="fade-up">
                            <h2 className="heading mb-4">Ready to Experience Comfort?</h2>
                            <p className="mb-4 text-muted">
                                Book your stay at {room.room_name} today and enjoy world-class hospitality with all the amenities you need for a perfect getaway.
                            </p>
                            <Link 
                                href={`/reservation?room=${room.id}`} 
                                className="btn btn-primary text-white py-3 px-5 mr-3"
                            >
                                Book Now
                            </Link>
                            <Link 
                                href="/rooms" 
                                className="btn btn-outline-primary py-3 px-5"
                            >
                                View All Rooms
                            </Link>
                        </div>
                    </div>
                </div>
            </section>
        </>
    );
}
