import HeroSection from "@user/HeroSection";
import { useForm, usePage, Head } from "@inertiajs/react";
import { useEffect } from "react";
import { toast, Toaster } from "react-hot-toast";

export default function Contact(){
    const { flash, turnstileSiteKey, siteinfo } = usePage().props;
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        email: '',
        phone: '',
        subject: '',
        message: '',
        turnstile_token: '',
    });

    // Handle flash messages
    useEffect(() => {
        if (flash?.success) {
            toast.success(flash.success.message || 'Message sent successfully!');
            reset();
        }
        if (flash?.error) {
            toast.error(flash.error.message || 'An error occurred');
        }
    }, [flash, reset]);

    // Load Turnstile widget
    useEffect(() => {
        const script = document.createElement('script');
        script.src = 'https://challenges.cloudflare.com/turnstile/v0/api.js';
        script.async = true;
        script.defer = true;
        document.body.appendChild(script);

        // Set up global callback
        window.turnstileCallback = (token) => {
            setData('turnstile_token', token);
        };

        return () => {
            if (document.body.contains(script)) {
                document.body.removeChild(script);
            }
            delete window.turnstileCallback;
        };
    }, [setData]);

    const handleTurnstileCallback = (token) => {
        setData('turnstile_token', token);
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        
        if (!data.turnstile_token) {
            toast.error('Please complete the CAPTCHA verification');
            return;
        }

        post('/api/contact', {
            preserveScroll: true,
            onSuccess: () => {
                toast.success('Message sent successfully! We will respond as soon as possible.');
                reset();
            },
            onError: (errors) => {
                if (errors.turnstile) {
                    toast.error('CAPTCHA verification failed. Please try again.');
                } else {
                    toast.error('Failed to send message. Please try again.');
                }
            }
        });
    };

    // Format WhatsApp numbers for display
    const formatWhatsApp = (whatsapp) => {
        if (!whatsapp) return '';
        // Split by comma if multiple numbers
        const numbers = whatsapp.split(',').map(num => num.trim());
        return numbers.join(', ');
    };

    // Format email for display (multiple emails)
    const formatEmails = (email) => {
        if (!email) return '';
        // Split by comma if multiple emails
        const emails = email.split(',').map(e => e.trim());
        return emails.join(', ');
    };

    return(
        <>
        <Head title="Contact Us" />
        <Toaster />
        <HeroSection heading="Contact" breadcrumbs={["Home", "Contact"]} />
        <section className="section contact-section" id="next" style={{ paddingBottom: 0 }}>
        <div className="container">
            <div className="row">
            <div className="col-md-7" data-aos="fade-up" data-aos-delay="100">
                
                <form onSubmit={handleSubmit} className="bg-white p-md-5 p-4 mb-5 border">
                <div className="row">
                    <div className="col-md-6 form-group">
                    <label htmlFor="name">Name</label>
                    <input 
                        type="text" 
                        id="name" 
                        className={`form-control ${errors.name ? 'is-invalid' : ''}`}
                        value={data.name}
                        onChange={(e) => setData('name', e.target.value)}
                        required
                    />
                    {errors.name && <div className="invalid-feedback">{errors.name}</div>}
                    </div>
                    <div className="col-md-6 form-group">
                    <label htmlFor="phone">Phone</label>
                    <input 
                        type="text" 
                        id="phone" 
                        className={`form-control ${errors.phone ? 'is-invalid' : ''}`}
                        value={data.phone}
                        onChange={(e) => setData('phone', e.target.value)}
                    />
                    {errors.phone && <div className="invalid-feedback">{errors.phone}</div>}
                    </div>
                </div>
            
                <div className="row">
                    <div className="col-md-12 form-group">
                    <label htmlFor="email">Email</label>
                    <input 
                        type="email" 
                        id="email" 
                        className={`form-control ${errors.email ? 'is-invalid' : ''}`}
                        value={data.email}
                        onChange={(e) => setData('email', e.target.value)}
                        required
                    />
                    {errors.email && <div className="invalid-feedback">{errors.email}</div>}
                    </div>
                </div>

                <div className="row">
                    <div className="col-md-12 form-group">
                    <label htmlFor="subject">Subject</label>
                    <input 
                        type="text" 
                        id="subject" 
                        className={`form-control ${errors.subject ? 'is-invalid' : ''}`}
                        value={data.subject}
                        onChange={(e) => setData('subject', e.target.value)}
                        required
                    />
                    {errors.subject && <div className="invalid-feedback">{errors.subject}</div>}
                    </div>
                </div>

                <div className="row mb-4">
                    <div className="col-md-12 form-group">
                    <label htmlFor="message">Write Message</label>
                    <textarea 
                        name="message" 
                        id="message" 
                        className={`form-control ${errors.message ? 'is-invalid' : ''}`}
                        cols="30" 
                        rows="8"
                        value={data.message}
                        onChange={(e) => setData('message', e.target.value)}
                        required
                    ></textarea>
                    {errors.message && <div className="invalid-feedback">{errors.message}</div>}
                    </div>
                </div>

                {/* Turnstile CAPTCHA */}
                <div className="row mb-3">
                    <div className="col-md-12">
                        {turnstileSiteKey && (
                            <div 
                                className="cf-turnstile" 
                                data-sitekey={turnstileSiteKey}
                                data-callback="turnstileCallback"
                                data-theme="light"
                            ></div>
                        )}
                        {errors.turnstile && <div className="text-danger small mt-1">{errors.turnstile}</div>}
                    </div>
                </div>

                <div className="row">
                    <div className="col-md-6 form-group">
                    <button 
                        type="submit" 
                        className="btn btn-primary text-white font-weight-bold"
                        disabled={processing}
                    >
                        {processing ? 'Sending...' : 'Send Message'}
                    </button>
                    </div>
                </div>
                </form>

            </div>
            <div className="col-md-5" data-aos="fade-up" data-aos-delay="200">
                <div className="row">
                <div className="col-md-10 ml-auto contact-info">
                    {siteinfo?.address && (
                        <p>
                            <span className="d-block font-weight-bold">Address:</span> 
                            <span>{siteinfo.address}</span>
                        </p>
                    )}
                    {siteinfo?.phone && (
                        <p>
                            <span className="d-block font-weight-bold">Phone/Fax:</span> 
                            <span>{siteinfo.phone}</span>
                        </p>
                    )}
                    {siteinfo?.whatsapp && (
                        <p>
                            <span className="d-block font-weight-bold">WhatsApp:</span> 
                            <span>{formatWhatsApp(siteinfo.whatsapp)}</span>
                        </p>
                    )}
                    {siteinfo?.email && (
                        <p>
                            <span className="d-block font-weight-bold">Email:</span> 
                            <span>{formatEmails(siteinfo.email)}</span>
                        </p>
                    )}
                </div>
                </div>
            </div>
            </div>
        </div>
        </section>

        {/* Map Section Begin */}
        {siteinfo?.map_embed_code && (
            <div className="map" style={{ marginTop: 0, marginBottom: 0 }}>
                <div dangerouslySetInnerHTML={{ __html: siteinfo.map_embed_code }} style={{ margin: 0, padding: 0, display: 'block' }} />
            </div>
        )}
        {/* Map Section End */}
        </>
    );
}
