import { Head, usePage, Link } from "@inertiajs/react";
import HeroSection from "@user/HeroSection";
import { formatPriceTriple } from "@/utils/priceFormatter";

export default function EventRoomDetail() {
    const { eventRoom } = usePage().props;

    // Default/placeholder data if no event room is provided
    const room = eventRoom || {
        id: 1,
        name: "Conference Room A",
        primary_photo: null,
        price_fbu: "50000.00",
        price_usd: "28.00",
        short_description: "A versatile conference room perfect for meetings, workshops, and presentations.",
        full_description: "Our Conference Room A offers a professional and comfortable setting for your business meetings, workshops, and presentations. Equipped with modern audio-visual equipment and flexible seating arrangements, this space can accommodate various group sizes and event types. The room features natural lighting, comfortable seating, and state-of-the-art presentation technology to ensure your event runs smoothly.",
        capacity: 50,
        layouts: ["Theater", "Classroom", "Boardroom", "U-Shape"],
        equipment: ["Projector", "Screen", "Microphones", "Whiteboard", "WiFi", "Air Conditioning"],
        gallery: [],
    };

    // Get primary photo or placeholder
    const getPrimaryPhoto = () => {
        if (room.primary_photo) {
            return `/storage/${room.primary_photo}`;
        }
        return 'https://placehold.co/800x571';
    };

    // Get secondary image (first from gallery or placeholder)
    const getSecondaryPhoto = () => {
        if (room.gallery && room.gallery.length > 0) {
            return `/storage/${room.gallery[0]}`;
        }
        return 'https://placehold.co/800x800';
    };

    // Get gallery items for Photo slider
    const getGalleryItems = () => {
        if (room.gallery && room.gallery.length > 0) {
            return room.gallery.map((photo, index) => ({
                image: `/storage/${photo}`,
                caption: `${room.name} - Image ${index + 1}`,
                alt: `${room.name} gallery image ${index + 1}`
            }));
        }

        // Return placeholders if no gallery
        return Array.from({ length: 4 }, (_, index) => ({
            image: `https://placehold.co/1123x700`,
            caption: `${room.name} - Image ${index + 1}`,
            alt: `Room gallery placeholder ${index + 1}`
        }));
    };

    const galleryItems = getGalleryItems();

    // Open lightbox for gallery images
    const openLightbox = (index) => {
        if (typeof window !== 'undefined' && window.$ && window.$.fancybox) {
            const items = galleryItems.map(item => ({
                src: item.image,
                opts: { caption: item.caption }
            }));
            window.$.fancybox.open(items, {}, index);
        }
    };

    // Format price display in triple currency (BIF | USD | EUR)
    const priceDisplay = formatPriceTriple(room.price_fbu, room.price_usd, room.price_eur);

    // Parse layouts and equipment if they're strings
    const layouts = Array.isArray(room.layouts) ? room.layouts : (room.layouts ? JSON.parse(room.layouts) : []);
    const equipment = Array.isArray(room.equipment) ? room.equipment : (room.equipment ? JSON.parse(room.equipment) : []);

    return (
        <>
            <Head title={room.name} />

            {/* Hero Section with Breadcrumbs */}
            <HeroSection heading={room.name} breadcrumbs={["Home", "Meetings & Events", room.name]} />

            {/* Welcome-Style Intro Section */}
            <section className="py-5 bg-light">
                <div className="container">
                    <div className="row align-items-center">
                        {/* Images - Right Side */}
                        <div className="col-md-12 col-lg-7 ml-auto order-lg-2 position-relative mb-5" data-aos="fade-up">
                            <img src={getPrimaryPhoto()} alt={room.name} className="img-fluid rounded" />
                        </div>

                        {/* Content - Left Side */}
                        <div className="col-md-12 col-lg-4 order-lg-1" data-aos="fade-up">
                            <h2 className="heading">{room.name}</h2>
                            <p className="mb-4">{room.short_description}</p>

                            {/* Price Display */}
                            {priceDisplay && (
                                <p className="mb-3">
                                    <span style={{ fontSize: '1.4rem', fontWeight: 'bold', color: '#c9a227' }}>
                                        {priceDisplay}
                                    </span>
                                    <span className="text-muted d-block mt-1"> / day</span>
                                </p>
                            )}

                            <p>
                                <Link href={`/reservation?event=${room.id}`} className="btn btn-primary text-white py-2 mr-3">Book Now</Link>
                                <span className="mr-3 font-family-serif"><em>or</em></span>
                                <a href="#room-gallery" className="text-uppercase letter-spacing-1">View Gallery</a>
                            </p>
                        </div>
                    </div>
                </div>
            </section>

            {/* Room Details Section */}
            <section className="section" id="room-details">
                <div className="container">
                    <div className="row justify-content-center text-center mb-5">
                        <div className="col-md-7">
                            <h2 className="heading" data-aos="fade-up">Room Details</h2>
                            <p data-aos="fade-up" data-aos-delay="100">
                                Everything you need to know about our {room.name}
                            </p>
                        </div>
                    </div>

                    <div className="row">
                        {/* Room Specifications */}
                        <div className="col-lg-8" data-aos="fade-up">
                            <div className="bg-white p-4 p-md-5 mb-5 border rounded">
                                {/* Quick Info Cards */}
                                <div className="row mb-4">
                                    {room.capacity && (
                                        <div className="col-6 col-md-3 mb-3">
                                            <div className="text-center p-3 border rounded">
                                                <span className="icon-users d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                                <small className="text-muted d-block">Capacity</small>
                                                <strong>{room.capacity}</strong>
                                            </div>
                                        </div>
                                    )}
                                    <div className="col-6 col-md-3 mb-3">
                                        <div className="text-center p-3 border rounded">
                                            <span className="icon-calendar d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                            <small className="text-muted d-block">Type</small>
                                            <strong>Event Room</strong>
                                        </div>
                                    </div>
                                    {layouts.length > 0 && (
                                        <div className="col-6 col-md-3 mb-3">
                                            <div className="text-center p-3 border rounded">
                                                <span className="icon-layout d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                                <small className="text-muted d-block">Layouts</small>
                                                <strong>{layouts.length}</strong>
                                            </div>
                                        </div>
                                    )}
                                    {equipment.length > 0 && (
                                        <div className="col-6 col-md-3 mb-3">
                                            <div className="text-center p-3 border rounded">
                                                <span className="icon-cogs d-block mb-2" style={{ fontSize: '2rem', color: '#c9a227' }}></span>
                                                <small className="text-muted d-block">Equipment</small>
                                                <strong>{equipment.length}</strong>
                                            </div>
                                        </div>
                                    )}
                                </div>

                                {/* Full Description */}
                                {room.full_description && (
                                    <div className="mb-4">
                                        <h4 className="mb-3">About This Room</h4>
                                        <p className="text-muted" style={{ lineHeight: '1.8' }}>
                                            {room.full_description}
                                        </p>
                                    </div>
                                )}

                                {/* Layouts */}
                                {layouts.length > 0 && (
                                    <div className="mb-4">
                                        <h4 className="mb-3">Available Layouts</h4>
                                        <div className="row">
                                            {layouts.map((layout, index) => (
                                                <div key={index} className="col-6 col-md-4 mb-2">
                                                    <div className="d-flex align-items-center">
                                                        <span className="icon-check text-success mr-2"></span>
                                                        <span>{layout}</span>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                )}

                                {/* Equipment */}
                                {equipment.length > 0 && (
                                    <div>
                                        <h4 className="mb-3">Equipment & Amenities</h4>
                                        <div className="row">
                                            {equipment.map((item, index) => (
                                                <div key={index} className="col-6 col-md-4 mb-2">
                                                    <div className="d-flex align-items-center">
                                                        <span className="icon-check text-success mr-2"></span>
                                                        <span>{item}</span>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Booking Sidebar */}
                        <div className="col-lg-4" data-aos="fade-up" data-aos-delay="100">
                            <div className="bg-white p-4 mb-4 border rounded sticky-top" style={{ top: '100px' }}>
                                <h4 className="mb-4 text-center">Book This Event Room</h4>

                                {/* Price Display */}
                                <div className="text-center mb-4 pb-4 border-bottom">
                                    {priceDisplay && (
                                        <div>
                                            <span style={{ fontSize: '1.8rem', fontWeight: 'bold', color: '#c9a227' }}>
                                                {priceDisplay}
                                            </span>
                                            <span className="text-muted d-block mt-1"> / day</span>
                                        </div>
                                    )}
                                </div>

                                {/* Room Highlights */}
                                <ul className="list-unstyled mb-4">
                                    {room.capacity && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            Capacity: {room.capacity} people
                                        </li>
                                    )}
                                    {layouts.length > 0 && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            {layouts.length} Layout Options
                                        </li>
                                    )}
                                    {equipment.length > 0 && (
                                        <li className="mb-2">
                                            <span className="icon-check text-success mr-2"></span>
                                            {equipment.length} Equipment Items
                                        </li>
                                    )}
                                    <li className="mb-2">
                                        <span className="icon-check text-success mr-2"></span>
                                        Professional AV Support
                                    </li>
                                    <li className="mb-2">
                                        <span className="icon-check text-success mr-2"></span>
                                        Flexible Scheduling
                                    </li>
                                </ul>

                                {/* Book Now Button */}
                                <Link
                                    href={`/reservation?event=${room.id}`}
                                    className="btn btn-primary btn-block text-white py-3 font-weight-bold"
                                    style={{ fontSize: '1.1rem' }}
                                >
                                    Reserve Now
                                </Link>

                                <p className="text-center text-muted mt-3 mb-0" style={{ fontSize: '0.85rem' }}>
                                    Free setup and breakdown included
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* Photo Gallery Section - Using Photo component structure */}
            <section className="section slider-section bg-light" id="room-gallery">
                <div className="container">
                    <div className="row justify-content-center text-center mb-5">
                        <div className="col-md-7">
                            <h2 className="heading" data-aos="fade-up">Room Gallery</h2>
                            <p data-aos="fade-up" data-aos-delay="100">
                                Take a closer look at our {room.name}. Browse through the images to see the room's features and ambiance.
                            </p>
                        </div>
                    </div>
                    <div className="row">
                        <div className="col-md-12">
                            <div className="home-slider major-caousel owl-carousel mb-5" data-aos="fade-up" data-aos-delay="200">
                                {galleryItems.map((item, index) => (
                                    <div key={index} className="slider-item">
                                        <img
                                            src={item.image}
                                            alt={item.alt}
                                            className="img-fluid"
                                            style={{ cursor: 'pointer' }}
                                            onClick={() => openLightbox(index)}
                                        />
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* CTA Section */}
            <section className="section">
                <div className="container">
                    <div className="row justify-content-center text-center">
                        <div className="col-md-8" data-aos="fade-up">
                            <h2 className="heading mb-4">Ready to Host Your Event?</h2>
                            <p className="mb-4 text-muted">
                                Book {room.name} today and enjoy professional event hosting with all the equipment and support you need for a successful gathering.
                            </p>
                            <Link
                                href={`/reservation?event=${room.id}`}
                                className="btn btn-primary text-white py-3 px-5 mr-3"
                            >
                                Book Now
                            </Link>
                            <Link
                                href="/meetings-events"
                                className="btn btn-outline-primary py-3 px-5"
                            >
                                View All Event Rooms
                            </Link>
                        </div>
                    </div>
                </div>
            </section>
        </>
    );
}

