/**
 * Standardized date formatting utilities
 * Format: "6 January 2026" - consistent across all views
 */

const monthNames = [
    'January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December'
];

/**
 * Format a date string or Date object to "d Month YYYY" format
 * @param {string|Date} date - Date to format
 * @returns {string} Formatted date (e.g., "6 January 2026")
 */
export const formatDate = (date) => {
    if (!date) return '';
    
    const dateObj = typeof date === 'string' ? new Date(date) : date;
    
    if (isNaN(dateObj.getTime())) return '';
    
    const day = dateObj.getDate();
    const month = monthNames[dateObj.getMonth()];
    const year = dateObj.getFullYear();
    
    return `${day} ${month} ${year}`;
};

/**
 * Format a datetime string to "d Month YYYY, HH:mm" format
 * @param {string|Date} datetime - Datetime to format
 * @returns {string} Formatted datetime (e.g., "6 January 2026, 14:30")
 */
export const formatDateTime = (datetime) => {
    if (!datetime) return '';
    
    const dateObj = typeof datetime === 'string' ? new Date(datetime) : datetime;
    
    if (isNaN(dateObj.getTime())) return '';
    
    const day = dateObj.getDate();
    const month = monthNames[dateObj.getMonth()];
    const year = dateObj.getFullYear();
    const hours = String(dateObj.getHours()).padStart(2, '0');
    const minutes = String(dateObj.getMinutes()).padStart(2, '0');
    
    return `${day} ${month} ${year}, ${hours}:${minutes}`;
};

/**
 * Format date for backend (YYYY-MM-DD)
 * @param {Date} date - Date object
 * @returns {string} Backend-compatible date format
 */
export const formatDateForBackend = (date) => {
    if (!date) return '';
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
};

/**
 * Format date in short format for compact display (d Mon)
 * @param {string|Date} date - Date to format
 * @returns {string} Short formatted date (e.g., "28 Dec")
 */
export const formatDateShort = (date) => {
    if (!date) return '';
    
    const dateObj = typeof date === 'string' ? new Date(date) : date;
    
    if (isNaN(dateObj.getTime())) return '';
    
    const monthNamesShort = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    const day = dateObj.getDate();
    const month = monthNamesShort[dateObj.getMonth()];
    
    return `${day} ${month}`;
};
